local Widget = require("widgets/widget")
local Image = require ("widgets/image")
local ImageButton = require ("widgets/imagebutton")
local Panel = require ("widgets/panel")
local ScrollPanel = require ("widgets/scrollpanel")
local Text = require ("widgets/text")
local MenuSteamUserGroupRow = require ("widgets/ftf/menusteamusergrouprow")

local easing = require "util.easing"

----------------------------------------------------------------------
-- A list of Steam friends the player can see and potentially join

local MenuSteamUserGroupsWidget = Class(Widget, function(self, width, height)
	Widget._ctor(self, "MenuSteamUserGroupsWidget")

	-- How often to refresh the list, in seconds
	self.refresh_period = 1
	self.refresh_time_remaining = 0
	self.connecting_time_min_UI_delay = 0.5 -- Wait these seconds before moving on after connecting

	self.width = width or 900
	self.height = height or 900
	self.padding = 60

	-- Based on the bg texture
	self.header_size = 160
	self.footer_size = 54
	self.left_size = 40
	self.right_size = 40
	self.scroll_right_padding = 20 -- So the scrollbar doesn't touch the right edge
	self.scroll_contents_width = self.width - self.left_size - self.right_size - 100 -- So they have enough padding

	self.bg = self:AddChild(Panel("images/ui_ftf_multiplayer/multiplayer_btn_friends.tex"))
		:SetName("Background")
		:SetNineSliceCoords(70, 180, 600, 190)
		:SetSize(self.width, self.height)

	self.title = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.MENU_BUTTON_TITLE))
		:SetName("Title")
		:SetGlyphColor(UICOLORS.BACKGROUND_DARK)
		:SetHAlign(ANCHOR_LEFT)
		:SetAutoSize(self.width - self.padding*2)
		:SetText(STRINGS.UI.STEAMUSERGROUPSWIDGET.TITLE)
		:LayoutBounds("left", "top", self.bg)
		:Offset(self.padding, -self.padding)

	self.scroll = self:AddChild(ScrollPanel())
		:SetName("Scroll")
		:SetSize(self.width - self.left_size - self.right_size - self.scroll_right_padding, self.height - self.header_size - self.footer_size)
		:SetVirtualMargin(50)
		:LayoutBounds("left", "top", self.bg)
		:Offset(self.left_size, -self.header_size)
	self.scroll_contents = self.scroll:AddScrollChild(Widget())
		:SetName("Scroll contents")

	self.empty_label = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT))
		:SetGlyphColor(UICOLORS.LIGHT_TEXT_DARK)
		:SetHAlign(ANCHOR_MIDDLE)
		:SetAutoSize(self.scroll_contents_width)
		:SetText(STRINGS.UI.STEAMUSERGROUPSWIDGET.EMPTY_LABEL)
		:LayoutBounds("center", "top", self.bg)
		:Offset(0, -self.header_size-40)
end)

-- Start and StopUpdating have to be called by the parent
function MenuSteamUserGroupsWidget:OnUpdate(dt)

	-- If there's focus within the widget, keep track of what row is focused
	local focus_widget = self:GetFocusForOwner()
	if focus_widget and self:IsAncestorOf(focus_widget) then
		if focus_widget ~= self.last_focus_row then

			-- Find what index on the list this is
			for idx, row in ipairs(self.scroll_contents.children) do
				if focus_widget == row then
					self.last_focus_row = focus_widget
					self.last_focus_index = idx
					break
				end
			end
		end
	elseif self.last_focus_index then
		-- If we lost focus, we shouldn't force focus onto the list after updating it
		self.last_focus_row = nil
		self.last_focus_index = nil
	end


	-- Update the list periodically, and restore focus afterwards
	self.refresh_time_remaining = math.max(0, self.refresh_time_remaining - dt)
	if self.refresh_time_remaining <= 0 then
		self.refresh_time_remaining = self.refresh_period
		self:RefreshUserGroupsList()
	end

end

function MenuSteamUserGroupsWidget:_getUserGroupGames()
	local sortedbygroup = {}
	local mygroups = TheNet:GetUserGroups()	-- Populate 'sorted' with all my groups:

	if mygroups and (table.count(mygroups) > 0) then
		for _i, usergroup_data in ipairs(mygroups) do
			local groupinfo = {}
			groupinfo.id = usergroup_data.id
			groupinfo.name = usergroup_data.name
			groupinfo.avatarfilename = usergroup_data.avatarfilename
			groupinfo.games = {}
			sortedbygroup[usergroup_data.id] = groupinfo
		end
	end

	local myregion, games = TheNet:GetUserGroupGamesPlayingThisGame()

	local myprogression = TheNet:GetProgression()

	--go through each available game
	for _i, game_data in pairs(games) do
		if game_data.SlotsRemaining > 0 then	-- only list games with spots available
			--go through each group that this game belongs to
			for _a, ugg in ipairs(game_data.UserGroups) do
				if sortedbygroup[ugg.id] then
					local stored_gamedata = {}
					stored_gamedata.id = game_data.LobbyID
					stored_gamedata.region = game_data.Region
					stored_gamedata.progression_delta = myprogression - game_data.Progression
					table.insert(sortedbygroup[ugg.id].games, stored_gamedata)
				end
			end
		end
	end

	local flatlist = {}
	for _, groupinfo in pairs(sortedbygroup) do
		if #groupinfo.games > 0 then
			table.insert(flatlist, groupinfo)
		end
	end

-- TODO: sort all the ones with 0 games to the bottom?
	table.sort(flatlist, function(a, b) return #a.games > #b.games end)

	return myregion, flatlist
end

function MenuSteamUserGroupsWidget:RefreshUserGroupsList()
	-- TODO: Instead of re-creating these, find the old button and update it instead

	-- Remove old buttons, if any
	self.scroll_contents:RemoveAllChildren()

	local myregion, groupsWithGames = self:_getUserGroupGames()

	-- Add widgets for each
	for _i, groupinfo in ipairs(groupsWithGames) do
		local row_button = self.scroll_contents:AddChild(MenuSteamUserGroupRow(self.scroll_contents_width, groupinfo))
		row_button:SetOnClickFn(function(device_type, device_id) self:OnClickJoinUserGroup(row_button, myregion, groupinfo, device_type, device_id) end)
	end





	-- Get a list of friends currently in game
--	local usergroups = TheNet:GetUserGroupGamesPlayingThisGame()

	-- Add widgets for each
--	for _i, groupinfo in ipairs(groupsWithGames) do
--		local row_button = self.scroll_contents:AddChild(MenuSteamUserGroupRow(self.scroll_contents_width, usergroup_data))
--		row_button:SetOnClickFn(function(device_type, device_id) self:OnClickJoinUserGroup(row_button, usergroup_data, device_type, device_id) end)
--	end

	-- Layout
	self.scroll_contents:LayoutChildrenInColumn(25, "left")
		:LayoutBounds("center", "top", 0, 0)

	-- Hide empty label
	self.empty_label:SetShown(not self.scroll_contents:HasChildren())

	self.scroll:RefreshView()


	-- If there was focus, restore it to the same index, or as close as possible
	if self.last_focus_index then
		self.last_focus_index = math.min(self.last_focus_index, #self.scroll_contents.children)

		-- Check if there are rows to give focus to
		if self.last_focus_index > 0 then

			-- Get the row currently in that index
			local row = self.scroll_contents.children[self.last_focus_index]

			-- Tell the screen to move the brackets to the new row without animating them in
			-- (in case they're in the same spot, the animating out and in looks odd)
			local show_immediately = true
			self:GetFE():SetFocusWidget(row, 1, show_immediately)

		else

			-- There are no rows to give focus to. Give focus to something else to the left
			local focus = self:OnFocusMove("left", TheInput:GetKeyboard())
			if focus then
				focus:SetFocus()
			end
		end
	end
end


local function GetEligibleServers(my_region, server_list)
	local chosen_list = {}
	local MAX_SERVERS = 5

	local remove_selected = function(server_list, chosen_list)
		for i, id in ipairs(chosen_list) do
			local idx
			for j, server in ipairs(server_list) do
				if server.id == id then
					idx = j
					break
				end
			end

			if idx ~= nil then
				table.remove(server_list, idx)
			end
		end
	end	

	local select_servers = function(filter_fn)
		--add eligible servers
		for i, server in ipairs(server_list) do
			if #chosen_list < MAX_SERVERS and filter_fn(server) then
				table.insert(chosen_list, server.id)
			end
		end

		--remove the chosen ones
		remove_selected(server_list, chosen_list)
	end

	TheLog.ch.Networking:printf("Looking for games. %d available, my region = %s ", #server_list, my_region)

	--first add everyone in my region that has delta +/- 1
	TheLog.ch.Networking:printf("Adding games in my region with similar progression.")
	local nrgames = #chosen_list
	select_servers( function(server) 
		return server.region == my_region and math.abs(server.progression_delta) <= 1 and not TheNet:HasRecentlyJoinedUserGroupGame(server.id)
	end )

	local diff = #chosen_list - nrgames
	if diff > 0 then
		TheLog.ch.Networking:printf("%d found.", diff)
	end

	if #chosen_list >= MAX_SERVERS then
		TheLog.ch.Networking:printf("Sending chosen list")
		return chosen_list
	end

	--if not enough, add everyone not in my region that has delta +/- 1
	TheLog.ch.Networking:printf("Adding games not in my region with similar progression.")
	nrgames = #chosen_list
	select_servers( function(server) 
		return server.region ~= my_region and math.abs(server.progression_delta) <= 1 and not TheNet:HasRecentlyJoinedUserGroupGame(server.id)
	end )

	diff = #chosen_list - nrgames
	if diff > 0 then
		TheLog.ch.Networking:printf("%d found.", diff)
	end


	if #chosen_list >= MAX_SERVERS then
		TheLog.ch.Networking:printf("Sending chosen list")
		return chosen_list
	end

	--if not enough, add everyone in my region
	TheLog.ch.Networking:printf("Adding games in my region.")
	nrgames = #chosen_list
	select_servers( function(server) 
		return server.region == my_region and not TheNet:HasRecentlyJoinedUserGroupGame(server.id)
	end )

	diff = #chosen_list - nrgames
	if diff > 0 then
		TheLog.ch.Networking:printf("%d found.", diff)
	end

	if #chosen_list >= MAX_SERVERS then
		TheLog.ch.Networking:printf("Sending chosen list.")
		return chosen_list
	end

	--if not enough, add everyone else
	TheLog.ch.Networking:printf("Adding all other games.")
	nrgames = #chosen_list
	select_servers( function(server) 
		return true
	end )

	diff = #chosen_list - nrgames
	if diff > 0 then
		TheLog.ch.Networking:printf("%d found.", diff)
	end

	TheLog.ch.Networking:printf("Sending chosen list.")
	return chosen_list	
end 

function MenuSteamUserGroupsWidget:OnClickJoinUserGroup(row_button, my_region, groupinfo, device_type, device_id)
	if table.count(groupinfo.games) > 0 then
		local eligibleservers = GetEligibleServers(my_region, groupinfo.games)
		if eligibleservers and #eligibleservers > 0 then
			-- Disable the button for the moment
			self.selected_row_button = row_button
			self.selected_row_button:Disable()

			-- Keep track of the selected lobby
			self.selected_lobby = eligibleservers[math.random(#eligibleservers)]
		

			-- Show a "Connecting..." popup
			self.connect_popup = ShowConnectingToGamePopup()

			self:RunUpdater(Updater.Series{

				-- Wait at least this amount of time before actually going into the game
				Updater.Wait(self.connecting_time_min_UI_delay),

				-- Actually start the game
				Updater.Do(function()
					self:HandleGameStart(device_type, device_id)
				end)
			})
		end
	end
	return self
end

function MenuSteamUserGroupsWidget:HandleGameStart(device_type, device_id)

	-- Start the game
	local inputID = TheInput:ConvertToInputID(device_type, device_id)
	TheNet:AddRecentlyJoinedUserGroupGame(self.selected_lobby)
	TheNet:StartGame(inputID, "invite", self.selected_lobby)
end

return MenuSteamUserGroupsWidget
